/*
 * Generate a single wirelist file from the database
 */
#include <unistd.h>
#include <stdio.h>

#include "db.h"
#include "libfma.h"
#include "lf_fabric.h"

lf_fabric_t *fp;
int do_write_xbar_links;

static void
usage()
{
  fprintf(stderr, "This reads the database of connections and prints out a list of the\n");
  fprintf(stderr, "contents of each switch's slots and where everything is connected.\n");
  fprintf(stderr, "Running and reviewing this as a first step after creating the\n");
  fprintf(stderr, "database is a good way to notice links that are out.\n");
  fprintf(stderr, "This is run on a node which has filesystem access to the database files.\n");
  fprintf(stderr, "Since this program does not modify the database, it can be run at any time.\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "Usage: fm_db2wirelist\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -x - show gory xbar detail\n");
  fprintf(stderr, "  -R <fms_run> - specify FMS_RUN directory\n");
  fprintf(stderr, "  -N <fms_db_name> - specify database name\n");
  fprintf(stderr, "  -V - print version\n");
  exit(1);
}

void
write_enclosures(
  FILE *wfp,
  lf_fabric_t *fp)
{
  int e;
  lf_enclosure_t *ep;

  fprintf(wfp, "[enclosures]\n");
  fprintf(wfp, "Enclosure,Type\n");
  for (e=0; e<fp->num_enclosures; ++e) {
    ep = fp->enclosures[e];
    fprintf(wfp, "%s,%s\n", ep->name, ep->product_id);
  }
  fprintf(wfp, "\n");
}

void
write_linecards(
  FILE *wfp,
  lf_fabric_t *fp)
{
  int e;

  fprintf(wfp, "[linecards]\n");
  fprintf(wfp, "Enclosure,Slot,Type\n");
  for (e=0; e<fp->num_enclosures; ++e) {
    lf_enclosure_t *ep;
    int l;

    ep = fp->enclosures[e];
    for (l=0; l<ep->num_lc_slots; ++l) {
      lf_linecard_t *lp;

      lp = ep->slots[l];
      if (lp == NULL) continue;

      fprintf(wfp, "%s,%d,%s\n", ep->name, lf_slot_display_no(lp),
	  		lp->product_id);
    }
  }
  fprintf(wfp, "\n");
}

void
write_switch_links(
  FILE *wfp,
  lf_fabric_t *fp)
{
  int e;

  fprintf(wfp, "[links]\n");
  for (e=0; e<fp->num_enclosures; ++e) {
    lf_enclosure_t *ep;
    int l;

    ep = fp->enclosures[e];
    for (l=0; l<ep->num_slots; ++l) {
      lf_linecard_t *lp;
      int x;

      lp = ep->slots[l];
      if (lp == NULL) continue;

      for (x=0; x<lp->num_xcvrs; ++x) {
        lf_xcvr_t *xp;
	lf_node_t *np;
	int rport;
	int p;

	xp = LF_XCVR(lp->xcvrs[x]);
	for (p=0; p<xp->num_conns; ++p) {
	
	  fprintf(wfp, "%s s%d p%d", ep->name, 
		lf_slot_display_no(lp),
		lp->xcvr_labels[x]);
	  if (xp->num_conns > 1) {
	    fprintf(wfp, ":%d", p);
	  }
	  fprintf(wfp, " -- ");

	  np = xp->ports[p];
	  rport = xp->rports[p];

	  if (np != NULL) {
	    if (np->ln_type == LF_NODE_LC_XCVR) {
	      lf_xcvr_t *xp2;
	      lf_linecard_t *lp2;

	      xp2 = LF_XCVR(np);
	      lp2 = xp2->p.linecard;

	      fprintf(wfp, "%s s%d p%d", lp2->enclosure->name, 
		    lf_slot_display_no(lp2),
		    lp2->xcvr_labels[xp2->port]);
	      if (xp2->num_conns > 1) {
		fprintf(wfp, ":%d", rport);
	      }
		      
	    } else if (np->ln_type == LF_NODE_NIC_XCVR) {
	      lf_xcvr_t *xp2;

	      xp2 = LF_XCVR(np);

	      /* subport means nothing for NICs yet */
	      fprintf(wfp, "%s n%d p%d", xp2->p.nic->host->hostname,
		      xp2->p.nic->host_nic_id, xp2->port);
	    }
	  }
	  fprintf(wfp, "\n");
	}
      }
    }
  }
  fprintf(wfp, "\n");
}

void
write_xbar_links(
  FILE *wfp,
  lf_fabric_t *fp)
{
  int e;

  fprintf(wfp, "[xbars]\n");
  fprintf(wfp, "Enclosure,Slot,Xbar,Port,Enclosure/Host,Slot/NIC,Port\n");
  for (e=0; e<fp->num_enclosures; ++e) {
    lf_enclosure_t *ep;
    int l;

    ep = fp->enclosures[e];
    for (l=0; l<ep->num_slots; ++l) {
      lf_linecard_t *lp;
      int x;

      lp = ep->slots[l];
      if (lp == NULL) continue;

      for (x=0; x<lp->num_xbars; ++x) {
        lf_xbar_t *xp;
	lf_node_t *np;
	int p;
	
	xp = LF_XBAR(lp->xbars[x]);

	for (p=0; p<xp->num_ports; ++p) {
	  np = xp->topo_ports[p];

	  fprintf(wfp, "%s,%d,%d,%d", ep->name,
	      lf_slot_display_no(lp), x, p);

	  if (np != NULL) {
	    if (np->ln_type == LF_NODE_NIC) {
	      lf_nic_t *nicp;
	      int port;

	      nicp = LF_NIC(np);
	      port = xp->topo_rports[p];

	      fprintf(wfp, ",%s,%d,%d,0", nicp->host->hostname,
		      nicp->host_nic_id, port);

	      if (xp->phys_ports[p] == np) {
		fprintf(wfp, " [DIRECT]");
	      }

	    } else if (np->ln_type == LF_NODE_XBAR) {
	      lf_xbar_t *xp2;
	      int port;

	      xp2 = LF_XBAR(np);
	      port = xp->topo_rports[p];

	      fprintf(wfp, ",%s,%d,%d,%d", xp2->linecard->enclosure->name,
		      lf_slot_display_no(xp2->linecard),
		      xp2->xbar_no, port);
	    } else {
	      fprintf(wfp, ",???");
	    }
	  }
	  fprintf(wfp, "\n");
	}
      }
    }
  }
  fprintf(wfp, "\n");
}

void
write_host_links(
  FILE *wfp,
  lf_fabric_t *fp)
{
  int h;

  fprintf(wfp, "[hosts]\n");

  for (h=0; h<fp->num_hosts; ++h) {
    lf_host_t *hp;
    int n;

    hp = fp->hosts[h];
    for (n=0; n<hp->num_nics; ++n) {
      lf_nic_t *nicp;
      int p;

      nicp = hp->nics[n];
      for (p=0; p<nicp->num_ports; ++p) {
        lf_node_t *np;
	lf_xcvr_t *xp;

	fprintf(wfp, "%s n%d p%d -- ", hp->hostname, nicp->host_nic_id, p);

	xp = LF_XCVR(nicp->phys_ports[p]);
	np = xp->ports[0];

	if (np != NULL && np->ln_type == LF_NODE_LC_XCVR) {
	  lf_xcvr_t *xp2;

	  xp2 = LF_XCVR(np);
	  fprintf(wfp, "%s s%d p%d", xp2->p.linecard->enclosure->name,
		lf_slot_display_no(xp2->p.linecard),
		xp2->p.linecard->xcvr_labels[xp2->port]);
	  if (xp2->num_conns > 1) {
	    fprintf(wfp, ":%d", xp->rports[0]);
	  }
	}
	fprintf(wfp, " (%s)\n", lf_fw_type_string(hp->fw_type));
      }
    }
  }
  fprintf(wfp, "\n");
}

void
write_wirelist(
  char *filename,
  lf_fabric_t *fp)
{
  FILE *wfp;

  if (filename != NULL) {
    wfp = fopen(filename, "w");
    if (wfp == NULL) {
      perror(filename);
      exit(1);
    }
  } else {
    wfp = stdout;
  }

  write_enclosures(wfp, fp);
  write_linecards(wfp, fp);
  write_switch_links(wfp, fp);
  if (do_write_xbar_links) write_xbar_links(wfp, fp);
  write_host_links(wfp, fp);
 
  if (filename != NULL) fclose(wfp);
}


int
main(
  int argc,
  char **argv)
{
  int c;
  extern char *optarg;
  char *db_name;
  char *fms_run;
  char *wirelistname;

  lf_init();

  /* defaults */
  wirelistname = NULL;
  fms_run = NULL;
  db_name = NULL;

  /* command line args */
  while ((c = getopt(argc, argv, "hH:R:N:w:xV")) != EOF) switch (c) {
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  case 'h':
    usage();
    break;
  case 'H':
    fprintf(stderr,
	"Please note: -H has been replaced with -R and is deprecated\n");
    /* FALLSTHROUGH */

  case 'R':
    fms_run = optarg;
    break;
  case 'N':
    db_name = optarg;
    break;
  case 'w':
    wirelistname = optarg;
    break;
  case 'x':
    do_write_xbar_links = 1;
    break;
  }

  /* set dirs - doesn't care if they are NULL */
  lf_simple_set_fabric_dirs(fms_run, db_name);

  /* load the fabric */
  fp = lf_simple_load_fabric();
  if (fp == NULL) {
    fprintf(stderr, "Error loading fabric.\n");
    exit(1);
  }

  /* output the wirelist */
  write_wirelist(wirelistname, fp);

  exit(0);
}
